#ifndef AHLGREN_FITNESS
#define AHLGREN_FITNESS

#include <vector>
#include <list>
#include <memory>

#include "global.h"
#include "parameters.h"
#include "program.h"

namespace lp {

	// Positive coverage type
	typedef std::list<Program::clause>::iterator posex_iterator;
	typedef std::list<Program::clause>::const_iterator negex_iterator;
	typedef std::vector<posex_iterator> pcover_type;
	typedef std::vector<std::pair<negex_iterator,double>> ncover_type;

	const long long invalid_fitness = numeric_limits<long long>::min();
	const long long worst_fitness = invalid_fitness + 1;

	// Class for best-so-far type
	class bsf_type {
	public:
		bsf_type() : f(invalid_fitness) {}
		bsf_type(const bitstring& bs) : msk(bs), f(invalid_fitness) {}
		bsf_type(bitstring&& bs) : msk(std::move(bs)), f(invalid_fitness) {}
		bsf_type(bsf_type&& b) : msk(std::move(b.msk)), posi(std::move(b.posi)), negi(std::move(b.negi)), f(b.f) {}
		bsf_type& operator=(bsf_type&& b) {
			if (this != &b) {
				msk = std::move(b.msk);
				posi = std::move(b.posi);
				negi = std::move(b.negi);
				f = b.f;
			}
			return *this;
		}
		// Clear
		void clear() { posi.clear(); negi.clear(); msk.clear(); f = invalid_fitness; } 
		// Is BSF set?
		bool is_nil() const { return f == invalid_fitness; }
		// Is candidate valid? (i.e. not mode/settings violated)
		bool is_valid() const { return !is_nil(); }
		void set_invalid() { f = invalid_fitness; }
		// Get Positive Coverage
		int pos() const { return posi.size(); }
		// Add Positive Example
		void add_pos(posex_iterator i) { posi.push_back(i); }
		// Get/Set Positive Example iterators
		pcover_type& pcover() { return posi; }
		const pcover_type& pcover() const { return posi; }
		void pcover(const pcover_type& pc) { posi = pc; }
		void pcover(pcover_type&& pc) { posi = std::move(pc); }
		// Get/Set Negative Coverage
		int neg() const { return negi.size(); }
		void add_neg(negex_iterator i, double c = 0) { negi.push_back(std::make_pair(i,c)); }
		ncover_type& ncover() { return negi; }
		const ncover_type& ncover() const { return negi; }
		void ncover(const ncover_type& nc) { negi = nc; }
		void ncover(ncover_type&& nc) { negi = std::move(nc); }
		// Check consistency
		bool is_consistent(int noise) const { return neg() <= noise; }
		// Get bitstring
		const bitstring& mask() const { return msk; }
		// Get body literals
		//int body_size() const { return std::count(msk.begin(),msk.end(),1); }
		// Set Fitness
		void fitness(long long d) { f = d; }
		long long fitness() const { return f; }
		int fcompare(const bsf_type& b, int noise) const {
			if (is_consistent(noise)) {
				if (!b.is_consistent(noise)) {
					// this > b
					return 1;
				}
			} else if (b.is_consistent(noise)) {
				// this < b
				return -1;
			}
			// both consistent or inconsistent
			if (fitness() < b.fitness()) return -1;
			else if (fitness() > b.fitness()) return 1;
			else return 0;
		}
		// Compare candidates
		bool operator==(const bsf_type& b) const { return msk == b.msk; }
		bool operator!=(const bsf_type& b) const { return !(*this == b); }
		void print(ostream& os) const {
			os << "(" << fitness() << "," << pos() << "," << neg() << "," 
				<< std::count(msk.begin(),msk.end(),1) << ") : ";
			std::for_each(msk.begin(),msk.end(),[&](int a){ os << a; });
			os << "\n";
		}
	protected:
		// Members
		bitstring msk;
		pcover_type posi; // positive coverage
		ncover_type negi;
		long long f; // fitness value

		bsf_type(int); // forbid
		bsf_type(unsigned); // forbid
		bsf_type(long long); // forbid
	};

	// Output bsf
	inline ostream& operator<<(ostream& os, const bsf_type& b) {
		b.print(os);
		return os;
	}



	// Helper: compute full coverage of positive and negative examples
	template <typename sol_type>
	inline void compute_negcov(
		const Functor& cand,
		sol_type& sol, // use copy of candidate solution to call mutable covers
		Program& kb, // knowledge base (to compute coverage)
		const list<Program::clause>& nex, // negative examples
		double& evalc, // number of examples evaluated
		deadline_t deadline)
	{
		if (evalc + nex.size() > kb.params.force_int(parameters::max_examples)) {
			throw search_aborted();
		}
		evalc += nex.size();

		const auto noise = kb.params.force_int(parameters::noise);
		const auto max_e = kb.params.force_int(parameters::max_examples);
		const auto reorder_negs = kb.params.force_int(parameters::reorder_negs);
		double time = 0;
		long long ic = 0;
		double* time_ptr = (reorder_negs == 2 ? &time : nullptr);
		// Insert candidate
		assert( cand.is_function() && cand.head()->is_function() );
		const sign_t sc = cand.head()->signature();
	    //std::cerr << "Covers, adding candidate: " << *cand << "\n";
		kb.push_back(cand);
		try {
			for (auto i = nex.begin(); i != nex.end(); ++i) {
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				if (++evalc > max_e) throw search_aborted();
				if (kb.covers(*i,reorder_negs == 1 : double(ic) : time, &deadline)) {
					sol.add_neg(i,time);
				}
			}
		} catch (...) {
			// Erase candidate before rethrowing
			kb.pop_back(sc);
			throw;
		}
		// Erase candidate
		kb.pop_back(sc);
		return true;
	}

	template <typename sol_type>
	inline bool compute_consistency(
		const Functor& cand, // candidate solution
		sol_type& sol,
		Program& kb, // knowledge base (to compute coverage)
		const list<Program::clause>& nex, // negative examples
		double& evalc, // number of examples evaluated
		deadline_t deadline)
	{
		const auto noise = kb.params.force_int(parameters::noise);
		const auto max_e = kb.params.force_int(parameters::max_examples);
		const auto reorder_negs = kb.params.force_int(parameters::reorder_negs);
		double time = 0;
		long long ic = 0;
		double* time_ptr = (reorder_negs == 2 ? &time : nullptr);
		// Insert candidate
		assert( cand.is_function() && cand.head()->is_function() );
		const sign_t sc = cand.head()->signature();
	    //std::cerr << "Covers, adding candidate: " << *cand << "\n";
		kb.push_back(cand);
		bool fail = false;
		try {
			for (auto i = nex.begin(); i != nex.end() && !fail; ++i) {
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				if (++evalc > max_e) throw search_aborted();
				if (kb.covers(*i,&ic,time_ptr,&deadline)) {
					sol.add_neg(i, reorder_negs == 1 ? double(ic) : time);
					if (!sol.is_consistent(noise)) {
						fail = true;
					}
				}
			}
		} catch (...) {
			// Erase candidate before rethrowing
			kb.pop_back(sc);
			throw;
		}
		// Erase candidate
		kb.pop_back(sc);
		return !fail;
	}

	template <typename sol_type>
	inline void compute_poscov(
		const Functor& cand,
		sol_type& sol, // use copy of candidate solution to call mutable covers
		Program& kb, // knowledge base (to compute coverage)
		const list<Program::clause>& pex, // positive examples
		double& evalc, // number of examples evaluated
		deadline_t deadline)
	{
		// Non-const access to pex as pcover needs non-const iterators
		const auto max_e = kb.params.force_int(parameters::max_examples);
		assert(cand.head() && cand.head()->is_function());
		const sign_t sc = cand.head()->signature();
		auto& pex_nc = const_cast<list<clause>&>(pex);
		kb.push_back(cand);
		try {
			for (auto i = pex_nc.begin(); i != pex_nc.end(); ++i) {
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				if (++evalc > max_e) throw search_aborted();
				if (kb.covers(*i,nullptr,nullptr,&deadline)) {
					sol.add_pos(i);
				}
			}
		} catch (...) {
			kb.pop_back(sc);
			throw;
		}
		kb.pop_back(sc);
	}

	template <typename sol_type>
	void compute_coverage(
		const Functor& cand,
		sol_type& sol, // use copy of candidate solution to call mutable covers
		Program& kb, // knowledge base (to compute coverage)
		const list<Program::clause>& pex, // positive examples
		const list<Program::clause>& nex, // negative examples
		double& evalc, // number of examples evaluated
		deadline_t deadline)
	{
		assert(sol.pcover().empty());
		// Fitness value
		//cerr << "Evaluating coverage: "; print_candidate(std::cerr,cand); cerr << "\n";
		const auto max_e = kb.params.force_int(parameters::max_examples);
		auto& pex_nc = const_cast<list<Program::clause>&>(pex);
		const auto reorder_negs = kb.params.force_int(parameters::reorder_negs);
		double time = 0;
		long long ic = 0;
		double* time_ptr = (reorder_negs == 2 ? &time : nullptr);

		assert(cand.head() && cand.head()->is_function());
		const sign_t sc = cand.head()->signature();
		kb.push_back(cand);

		try {
			//std::cerr << "compute negative coverage...\n";
			for (auto i = nex.begin(); i != nex.end(); ++i) {
				assert( !decompile(*i).head() );
				assert( i->back().opcode == instruction::PROMPT );
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				if (++evalc > max_e) throw search_aborted();
				//std::cerr << "  neg: " << decompile(*i) << "...";
				if (kb.covers(*i,&ic,time_ptr,&deadline)) {
					sol.add_neg(i,reorder_negs == 1 ? double(ic) : time);
					//std::cerr << "  Negative covered " << ++k << ": " << decompile(*i) << "\n";
				}
				//std::cerr << "done\n";
			}
			//std::cerr << "compute positive coverage...\n";
			for (auto i = pex_nc.begin(); i != pex_nc.end(); ++i) {
				assert( !decompile(*i).head() );
				assert( i->back().opcode == instruction::PROMPT );
				if (std::chrono::steady_clock::now() > deadline) throw time_out();
				if (++evalc > max_e) throw search_aborted();
				if (kb.covers(*i,nullptr,nullptr,&deadline)) {
					//std::cerr << "  Positive covered " << ++k << ": " << decompile(*i) << "\n";
					sol.add_pos(i);
				}
			}
			//std::cerr << "coverage DONE...\n";
		} catch (...) {
			DEBUG_TRACE( cerr << "caught error\n" );
			kb.pop_back(sc);
			throw;
		}
		kb.pop_back(sc);
		DEBUG_TRACE( cerr << "evaluation done\n" );
	}

	// Define Function Object for fitness evaluation
	template <typename sol_type>
	class fitness_function {
	public:
		// fitness_function() : tlevel(0) { }
		fitness_function(int t) : tlevel(t) { }
		virtual void init(const Functor&, const vector<Mode>&, Program&, 
			const list<Program::clause>&, const list<Program::clause>&, double&) { }
		virtual void evaluate(const Functor& cand, sol_type& sol, Program& kb, const list<Program::clause>& pex,
			const list<Program::clause>& nex, const sol_type&, double& evalc, deadline_t deadline) = 0;
		int termination() const { return tlevel; }
	protected:
		int tlevel; // termination level
	};


	// Compression Measure: P-N-L-Estimate
	// Variable depths
	map<id_type,int> var_depth(const Functor& f, const io_map& iov);
	// Optimistic estimate of body literals needed
	int optimistic_estimate(const Functor& cand, const io_map& iovars);

	// Fitness compression function object
	template <typename sol_type>
	struct fitness_compression : public fitness_function<sol_type> {
		// fitness_compression() : fitness_function() {}
		fitness_compression(int l) : fitness_function(l) {}
		void init(const Functor&, const vector<Mode>&, Program&, 
			const list<Program::clause>&, const list<Program::clause>&, double&);
		void evaluate(const Functor& cand, sol_type& sol, Program& kb, const list<Program::clause>& pex,
			const list<Program::clause>& nex, const sol_type&, double& evalc, deadline_t deadline);
	protected:
		int opt_est; // optimal estimate
	};


	template <typename sol_type>
	struct fitness_lex : public fitness_function<sol_type> {
		// fitness_lex() : fitness_function() {}
		fitness_lex(int l) : fitness_function(l) {}
		void evaluate(const Functor& cand, sol_type& sol, Program& kb, const list<Program::clause>& pex,
			const list<Program::clause>& nex, const sol_type&, double& evalc, deadline_t deadline);
	};


	//// Learn from positive data only using generality estimate
	//// Fitness posonly function object
	//template <typename sol_type>
	//struct fitness_posonly : public fitness_function<sol_type> {
	//	// fitness_posonly() : fitness_function() {}
	//	fitness_posonly(int l) : fitness_function(l) {}
	//	void init(const Functor&, const vector<Mode>&, Program&, 
	//		const list<Program::clause>&, const list<Program::clause>&, double&);
	//	void evaluate(const Functor& cand, sol_type& sol, Program& kb, const list<Program::clause>& pex,
	//		const list<Program::clause>& nex, const sol_type&, double& evalc);
	//	// Get info
	//	int sample_size() const { return int(samples.size()); }
	//	int sample_cov() const { return scovj; }
	//	int example_cov() const { return kbcovj; }
	//protected:
	//	list<Program::clause> samples;
	//	int scovj; // sample coverage
	//	int kbcovj; // KB example coverage
	//};




	//============================ Member Function Definitions =============================//


	template <typename sol_type>
	void fitness_compression<sol_type>::init(
		const Functor& cand, 
		const vector<Mode>& modes, 
		Program&, 
		const list<Program::clause>&, 
		const list<Program::clause>&,
		double&) 
	{
		// Compute optimistic estimate
		const auto iovars = make_io_map(cand,modes);
		opt_est = optimistic_estimate(cand,iovars);
		if (opt_est == numeric_limits<int>::max()) {
			throw bottom_rejected(); // not output connected
		}
	}

	template <typename sol_type>
	void fitness_compression<sol_type>::evaluate(
		const Functor& cand,
		sol_type& sol, 
		Program& kb, 
		const list<Program::clause>& pex, 
		const list<Program::clause>& nex, 
		const sol_type&, 
		double& evalc,
		deadline_t deadline) 
	{
		// Skip full evaluation only if explicitly set to partial
		if (kb.params.is_set(parameters::evalpartial)) {
			// Do partial fitness evaluation
			// Check consistency
			if (!compute_consistency(cand,sol,kb,nex,evalc,deadline)) {
				sol.fitness( worst_fitness);
				return;
			}
			// Consistent: compute positive coverage
			compute_poscov(cand,sol,kb,pex,evalc,deadline);
		} else {
			// Default: do full fitness evaluation
			compute_coverage(cand,sol,kb,pex,nex,evalc,deadline);
		}
		// Set compression = P - N - L - opt_est
		sol.fitness( 1000 * (sol.pos() - sol.neg() - std::count(sol.mask().begin(),sol.mask().end(),1) - opt_est) );
		const bool rec_penalty = kb.params.is_set(parameters::recursion_penalty);
		if (rec_penalty) {
			const sign_t hsign = cand.head()->signature();
			const int rec_lits = std::count_if(cand.body_begin(),cand.body_end(),[&](const Functor& l){ return l.sign(hsign.first,hsign.second); });
			sol.fitness( sol.fitness() - rec_lits );
		}
	}


	//template <typename sol_type>
	//inline void fitness_strict<sol_type>::evaluate(
	//	sol_type& sol, 
	//	Program& kb, 
	//	const list<Program::clause>& pex, 
	//	const list<Program::clause>& nex, 
	//	const sol_type&, 
	//	double& evalc)
	//{
	//	compute_coverage(sol,kb,pex,nex,evalc);
	//	sol.fitness( 10000.0 * (sol.pos()-sol.neg()) - std::count(body_size) );
	//}


	template <typename sol_type>
	void fitness_lex<sol_type>::evaluate(
		const Functor& cand,
		sol_type& sol, // candidate solution
		Program& kb, // knowledge base (to compute coverage)
		const list<Program::clause>& pex, // positive examples
		const list<Program::clause>& nex, // negative examples
		const sol_type& bsf, // best so far fitness value
		double& evalc, // number of examples evaluated
		deadline_t deadline) 
	{
		assert(sol.pcover().empty() && sol.pos() == 0 && sol.neg() == 0);

		const auto max_e = kb.params.force_int(parameters::max_examples);
		// Default is partial, only do strict evaluation if explicitly set to false
		if (kb.params.is_unset(parameters::evalpartial)) {
			// Strict evaluation requested
			compute_coverage(cand,sol,kb,pex,nex,evalc,deadline);
		} else {
			// Default: partial evaluation
		    //std::cerr << "Evaluating fitness, consistent? ";
			if (!compute_consistency(cand,sol,kb,nex,evalc,deadline)) {
				// Inconsistent
			    //std::cerr << "Inconsistent\n";
				sol.fitness(worst_fitness);
				return;
			}
			// Consistent
		    //std::cerr << "Consistent\n";
			// Non-const access to pex as pcover needs non-const iterators
			// Note: if bsf is inconsistent, bsf.pos() == 0 so this works then too
			int left = pex.size();
			// Insert candidate
			assert( cand.is_function() && cand.head()->is_function() );
			const sign_t sc = cand.head()->signature();
			//std::cerr << "Covers, adding candidate: " << *cand << "\n";
			kb.push_back(cand);
			try {
				bool fail = false;
				auto& pex_nc = const_cast<list<Program::clause>&>(pex);
				for (auto i = pex_nc.begin(); i != pex_nc.end() && !fail; ++i) {
					if (std::chrono::steady_clock::now() > deadline) throw time_out();
					// terminate if L < pos(bsf) - pos(x)
					if (left < bsf.pos() - sol.pos()) {
						// fail with worst consistent (note that neg <= noise)
						sol.fitness(worst_fitness);
						fail = true;
					}
					// compute coverage and positive examples left
					if (++evalc > max_e) throw search_aborted();
					//std::cerr << "Computing positive coverage for: " << *cand << "\n";
					if (kb.covers(*i,nullptr,nullptr,&deadline)) {
						//std::cerr << "candidate covers: " << decompile(*i) << "\n";
						sol.add_pos(i);
					} else {
						//std::cerr << "candidate does NOT cover: " << decompile(*i) << "\n";
					}
					--left;
				}
			} catch (...) {
				kb.pop_back(sc);
				throw;
			}
			kb.pop_back(sc);
		}

		// Fitness value
		//DEBUG_TRACE( cerr << "Evaluating candidate: " << cand << " , bsf: " << bsf << "\n" );
		sol.fitness( 1000*(10000 * (sol.pos()-sol.neg()) - std::count(sol.mask().begin(),sol.mask().end(),1)) );
		const bool rec_penalty = kb.params.is_set(parameters::recursion_penalty);
		if (rec_penalty) {
			const sign_t hsign = cand.head()->signature();
			const int rec_lits = std::count_if(cand.body_begin(),cand.body_end(),[&](const Functor& l){ return l.sign(hsign.first,hsign.second); });
			sol.fitness( sol.fitness() - rec_lits );
		}
		return;
	}


	//template <typename sol_type>
	//void fitness_posonly<sol_type>::init(
	//	const Functor& cand,
	//	const vector<Mode>& modes,
	//	Program& thy, 
	//	const list<Program::clause>& pex,
	//	const list<Program::clause>& nex,
	//	double& evalc)
	//{
	//	// Update labels in thy to get probability distribution over the examples
	//	thy.normalize_labels();
	//	//cerr << "All labels should be 1:\n" << thy << "\n";
	//	// Update Labels by querying all examples for types
	//	std::unique_ptr<Functor> gcl = modes.find(*cand.head())->second.generating_clause();
	//    //std::cerr << "Generating clause: " << gcl << "\n";
	//	auto gen_at = thy.push_back(move(gcl));
	//	Program::qconstraints qc(thy.params);
	//	qc.update_labels = true;
	//	Program::answer_type ans;
	//	evalc += pex.size(); // increase evaluated example counter
	//	for_each(pex.begin(),pex.end(),[&](const Functor& e){
	//		ans.clear();
	//		thy.query(e,ans,false,qc);
	//	});
	//	thy.erase(gen_at); // remove generating clause
	//    //std::cerr << "Updated labels in KB:\n" << thy << "\n";
	//	// Create samples based on this cand clause and its modeh
	//	samples.clear();
	//	const auto sample_size = thy.params.force_int(parameters::psamples);
	//	samples.reserve(sample_size);
	//	for (int k = 0; k < sample_size; ++k) { // UPDATE: use generate_n
	//		// TODO: sample()
	//		//std::unique_ptr<Functor> smpl = thy.sample(modes.find(*cand.head())->second,thy.params);
	//		std::unique_ptr<Functor> smpl;
	//		if (smpl.is_nil() || !smpl.is_ground()) {
	//		    std::cerr << "Error: sample is non-ground: " << smpl << "\n";
	//			throw destruct_and_exit();
	//		}
	//		samples.push_back(move(smpl));
	//		assert( samples.back().is_ground() );
	//	}
	//    //std::cerr << "Generated " << samples.size() << " samples:\n";
	//	for (decltype(samples.size()) k = 0; k < samples.size(); ++k) { // UPDATE: use generate_n
	//	    //std::cerr << "Sample " << k+1 << ": " << samples[k] << "\n";
	//	}

	//	// Compute sample coverage ( for H(i-1) )
	//	evalc += samples.size(); // update example counter
	//	scovj = get<0>( thy.covers(samples.begin(),samples.end()) );
	//	// Compute example coverage (for p)
	//	evalc += pex.size();
	//	kbcovj = get<0>(thy.covers(pex.begin(),pex.end()));
	//}


	//template <typename sol_type>
	//void fitness_posonly<sol_type>::evaluate(
	//	const Functor& cand,
	//	sol_type& sol, 
	//	Program& kb,
	//	const list<Program::clause>& pex, 
	//	const list<Program::clause>& nex, 
	//	const sol_type&, 
	//	double& evalc)
	//{
	//	// Discard top element
	//	if (!cand.body()) {
	//		sol.fitness(worst_fitness);
	//		return;
	//	}

	//	// Copy candidate for faster evaluation
	//	
	//	const auto max_e = kb.params.force_int(parameters::max_examples);
	//	cerr << "Evaluating candidate: " << cand << "\n";

	//	// Compute sample coverage for gHi
	//	evalc += samples.size();
	//	sol.sample() = get<0>(kb.covers(cand,samples.begin(),samples.end()));
	//	// Generality g(Hi) = (c+1)/(s+2)
	//	const double gHi = (sol.sample() + 1.0)/(samples.size() + 2.0);
	//	const double gHj = (scovj + 1.0)/(samples.size() + 2.0);
	//	cerr << "g(Hi)   = " << gHi << "\n";
	//	cerr << "g(Hi-1) = " << gHj << "\n";

	//	// Compute example coverage 
	//	// Non-const access to pex as pcover needs non-const iterators
	//	compute_poscov(cand,sol,kb,pex,evalc);
	//	
	//	const double p = sol.pos() - kbcovj;
	//	if (p <= 0) {
	//		sol.fitness( worst_fitness );
	//		return;
	//	}
	//	const double iprop = pex.size() / p;
	//	cerr << "p/m = " << p << "/" << pex.size() << " = " << 1.0/iprop << "\n";
	//	// g(H) = m/p * ( g(Hi) - g(Hj) )
	//	// if gHi = gHj, we do g(H) as if gHi-gHj= (1 example diff) / (s+2)
	//	const double gH = iprop * (gHi > gHj ?  gHi-gHj : 1.0/(samples.size()+2.0));

	//	cerr << "g(H) = " << gH << "\n";
	//	// H's fitness = m * ln(1/g(H)) - |H|, where |H| = m/p * |C|
	//	sol.fitness( double(pex.size()) * std::log(1.0/gH) - iprop * cand.size() );
	//	cerr << "ln(P(H|E))-Dm = " << sol.fitness() << "\n";
	//}



	//=================== Define Fitness Evaluations ===================//

	template <typename sol_type>
	inline unique_ptr<fitness_function<sol_type>>
		pick_fitness_function(const parameters& pa)
	{
		const data& dat = pa[parameters::evalfn];
		if (!dat.is_atom()) {
			DEBUG_WARNING(cerr << "warning: invalid fitness function " << dat << "\n");
			return nullptr;
		}
		const int termval = pa.force_int(parameters::terminate);
		unique_ptr<fitness_function<sol_type>> fitness;
		if (strcmp(dat.get_atom(),"auto") == 0) {
			fitness.reset(nullptr);
		} else if (strcmp(dat.get_atom(),"compression") == 0) {
			fitness.reset(new fitness_compression<sol_type>(termval));
		} else if (strcmp(dat.get_atom(),"lex") == 0) { // efficient lex
			fitness.reset(new fitness_lex<sol_type>(termval));
		} else {
			DEBUG_WARNING(cerr << "warning: invalid fitness function " << dat << "\n");
		}
		return fitness;
	}

	//template <typename sol_type>
	//inline unique_ptr<fitness_posonly<sol_type>>
	//	pick_fitness_posonly(const parameters& pa)
	//{
	//	// Only one supported positive only function for now
	//	//const data& dat = pa[parameters::evalfn];
	//	// if (!dat.is_atom()) return nullptr;
	//	return unique_ptr<fitness_posonly<sol_type>>(new fitness_posonly<pbsf_type>(pa.force_int(parameters::terminate)));
	//}



}


#endif



